<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Mustache helper render pix icons.
 *
 * @package    core
 * @category   output
 * @copyright  2015 Damyon Wiese
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace core\output;

use Mustache_LambdaHelper;
use renderer_base;

/**
 * This class will call pix_icon with the section content.
 *
 * @copyright  2015 Damyon Wiese
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @since      2.9
 */
class mustache_pix_helper {

    /** @var renderer_base $renderer A reference to the renderer in use */
    private $renderer;

    /**
     * Save a reference to the renderer.
     * @param renderer_base $renderer
     */
    public function __construct(renderer_base $renderer) {
        $this->renderer = $renderer;
    }

    /**
     * Read a pix icon name from a template and get it from pix_icon.
     *
     * {{#pix}}t/edit,component,Anything else is alt text{{/pix}}
     *
     * The args are comma separated and only the first is required.
     *
     * @param string $text The text to parse for arguments.
     * @param Mustache_LambdaHelper $helper Used to render nested mustache variables.
     * @return string
     */
    public function pix($text, Mustache_LambdaHelper $helper) {
        // Split the text into an array of variables.
        $key = strtok($text, ",");
        $key = trim($key);
        $key = $helper->render($key);
        $component = strtok(",");
        $component = trim($component);
        if (!$component) {
            $component = '';
        }
        $component = $helper->render($component);

        $text = strtok("");
        $text = $helper->render($text);
        $data = json_decode(trim($text), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $data = array('alt' => format_string($text));
            if ($data['alt'] !== '') {
                $data['title'] = $data['alt'];
            }
        }


        if ($component === 'flexicon') {
            // TOTARA HACK: The pix helper has received a flex icon, we can use the flex_helper.
            // Arguments are passed as following:
            //    identifier, component, json encoded icon data.
            // This hack is facilitate by code in \core\output\flex_icon::export_for_pix()
            $flexicon = new flex_icon($key, $data);
            return $this->renderer->render($flexicon);
        }

        return trim($this->renderer->pix_icon($key, '', $component, $data));
    }
}

