<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Steps definitions for the upload repository type.
 *
 * @package    repository_upload
 * @category   test
 * @copyright  2013 David Monllaó
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

// NOTE: no MOODLE_INTERNAL test here, this file may be required by behat before including /config.php.

require_once(__DIR__ . '/../../../../lib/behat/behat_files.php');

use Behat\Mink\Exception\ExpectationException as ExpectationException,
    Behat\Gherkin\Node\TableNode as TableNode;

/**
 * Steps definitions to deal with the upload repository.
 *
 * Extends behat_files rather than behat_base as is file-related.
 *
 * @package    repository_upload
 * @category   test
 * @copyright  2013 David Monllaó
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class behat_repository_upload extends behat_files {

    /**
     * Uploads a file to the specified filemanager leaving other fields in upload form default. The paths should be relative to moodle codebase.
     *
     * @When /^I upload "(?P<filepath_string>(?:[^"]|\\")*)" file to "(?P<filemanager_field_string>(?:[^"]|\\")*)" filemanager$/
     * @throws ExpectationException Thrown by behat_base::find
     * @param string $filepath
     * @param string $filemanagerelement
     */
    public function i_upload_file_to_filemanager($filepath, $filemanagerelement) {
        \behat_hooks::set_step_readonly(false);
        $this->upload_file_to_filemanager($filepath, $filemanagerelement, new TableNode(array()), false);
    }

    /**
     * Uploads a file to the specified filemanager leaving other fields in upload form default and confirms to overwrite an existing file. The paths should be relative to moodle codebase.
     *
     * @When /^I upload and overwrite "(?P<filepath_string>(?:[^"]|\\")*)" file to "(?P<filemanager_field_string>(?:[^"]|\\")*)" filemanager$/
     * @throws ExpectationException Thrown by behat_base::find
     * @param string $filepath
     * @param string $filemanagerelement
     */
    public function i_upload_and_overwrite_file_to_filemanager($filepath, $filemanagerelement) {
        \behat_hooks::set_step_readonly(false);
        $this->upload_file_to_filemanager($filepath, $filemanagerelement, new TableNode(array()),
                get_string('overwrite', 'repository'));
    }

    /**
     * Uploads a file to the specified filemanager and confirms to overwrite an existing file. The paths should be relative to moodle codebase.
     *
     * @When /^I upload "(?P<filepath_string>(?:[^"]|\\")*)" file to "(?P<filemanager_field_string>(?:[^"]|\\")*)" filemanager as:$/
     * @throws ExpectationException Thrown by behat_base::find
     * @param string $filepath
     * @param string $filemanagerelement
     * @param TableNode $data Data to fill in upload form
     */
    public function i_upload_file_to_filemanager_as($filepath, $filemanagerelement, TableNode $data) {
        \behat_hooks::set_step_readonly(false);
        $this->upload_file_to_filemanager($filepath, $filemanagerelement, $data, false);
    }

    /**
     * Uploads a file to the specified filemanager. The paths should be relative to moodle codebase.
     *
     * @When /^I upload and overwrite "(?P<filepath_string>(?:[^"]|\\")*)" file to "(?P<filemanager_field_string>(?:[^"]|\\")*)" filemanager as:$/
     * @throws ExpectationException Thrown by behat_base::find
     * @param string $filepath
     * @param string $filemanagerelement
     * @param TableNode $data Data to fill in upload form
     */
    public function i_upload_and_overwrite_file_to_filemanager_as($filepath, $filemanagerelement, TableNode $data) {
        \behat_hooks::set_step_readonly(false);
        $this->upload_file_to_filemanager($filepath, $filemanagerelement, $data,
                get_string('overwrite', 'repository'));
    }

    /**
     * Uploads a file to filemanager
     *
     * @throws ExpectationException Thrown by behat_base::find
     * @param string $filepath Normally a path relative to $CFG->dirroot, but can be an absolute path too.
     * @param string $filemanagerelement
     * @param TableNode $data Data to fill in upload form
     * @param false|string $overwriteaction false if we don't expect that file with the same name already exists,
     *     or button text in overwrite dialogue ("Overwrite", "Rename to ...", "Cancel")
     */
    protected function upload_file_to_filemanager($filepath, $filemanagerelement, TableNode $data, $overwriteaction = false) {
        global $CFG;

        // TOTARA: file upload does not work much, we use auxiliary behat filesystem repository instead

        // Attaching specified file to the node.
        // Replace 'admin/' if it is in start of path with $CFG->admin .
        if (substr($filepath, 0, 6) === 'admin/') {
            $filepath = $CFG->dirroot . DIRECTORY_SEPARATOR . $CFG->admin .
                    DIRECTORY_SEPARATOR . substr($filepath, 6);
        }
        $filepath = str_replace('/', DIRECTORY_SEPARATOR, $filepath);
        if (!is_readable($filepath)) {
            $filepath = $CFG->dirroot . DIRECTORY_SEPARATOR . $filepath;
            if (!is_readable($filepath)) {
                throw new ExpectationException('The file to be uploaded does not exist.', $this->getSession());
            }
        }

        $filename = basename($filepath);
        $behatpath = "$CFG->dataroot/repository/behat/$filename";
        @mkdir(dirname($behatpath), 02777, true);
        @unlink($behatpath);
        copy($filepath, $behatpath);
        chmod($behatpath, 0666);

        $filemanagernode = $this->get_filepicker_node($filemanagerelement);

        // Opening the select repository window and selecting the upload repository.
        $this->open_add_file_window($filemanagernode, 'behat');

        // Ensure all the form is ready.
        $noformexception = new ExpectationException('The filesystem file form is not ready', $this->getSession());
        $fileelement = $this->find(
            'xpath',
            "//div[contains(concat(' ', normalize-space(@class), ' '), ' file-picker ')]" .
                "[contains(concat(' ', normalize-space(@class), ' '), ' repository_filesystem ')]" .
                "/descendant::div[@class='fp-content']" .
            "//p[contains(., '$filename')]"
            ,
            $noformexception
        );
        // After this we have the elements we want to interact with.

        $fileelement->click();
        $this->wait_for_pending_js();

        // Fill the form in Upload window.
        $datahash = $data->getRowsHash();

        // The action depends on the field type.
        foreach ($datahash as $locator => $value) {

            $field = behat_field_manager::get_form_field_from_label($locator, $this);

            // Delegates to the field class.
            $field->set_value($value);
        }
        $this->wait_for_pending_js();

        // Submit the file.
        $submit = $this->find_button(get_string('getfile', 'repository'));
        $submit->press();

        // We wait for all the JS to finish as it is performing an action.
        $this->getSession()->wait(self::TIMEOUT, self::PAGE_READY_JS);

        if ($overwriteaction !== false) {
            $overwritebutton = $this->find_button($overwriteaction);
            $this->ensure_node_is_visible($overwritebutton);
            $overwritebutton->click();

            // We wait for all the JS to finish.
            $this->getSession()->wait(self::TIMEOUT, self::PAGE_READY_JS);
        }

        $this->wait_for_pending_js();
    }

}
